<?php
declare(strict_types=1);
/**
 * dailyh-horoscope-new.php (v2)
 * Iframe-only endpoint: renders a single sign's daily horoscope.
 * - Case-insensitive sign matching
 * - No-cache headers
 * - Auto-resize postMessage
 */

@header('Content-Type: text/html; charset=utf-8');
@header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
@header('Pragma: no-cache');
@header('Expires: 0');
@header('X-Robots-Tag: noindex, nofollow');

mb_internal_encoding('UTF-8');
date_default_timezone_set('Asia/Kolkata');

function safe_get(string $key, string $default = ''): string {
    if (!isset($_GET[$key])) return $default;
    $val = trim((string)$_GET[$key]);
    $val = preg_replace('/[^a-z0-9\-\_\/\.]/i', ' ', $val);
    $val = preg_replace('/\s+/', ' ', $val);
    return $val === '' ? $default : $val;
}
function normalize_sign(string $raw): string {
    $raw = strtolower($raw);
    $map = [
        'aries'=>'Aries','taurus'=>'Taurus','gemini'=>'Gemini','cancer'=>'Cancer',
        'leo'=>'Leo','virgo'=>'Virgo','libra'=>'Libra','scorpio'=>'Scorpio',
        'sagittarius'=>'Sagittarius','capricorn'=>'Capricorn','aquarius'=>'Aquarius','pisces'=>'Pisces',
        'sagi'=>'Sagittarius','sag'=>'Sagittarius','cap'=>'Capricorn','aqu'=>'Aquarius','pis'=>'Pisces',
    ];
    return $map[$raw] ?? 'Aries';
}
function normalize_date(?string $raw): string {
    if (!$raw) return date('Y-m-d');
    $t = strtotime($raw);
    if ($t === false) return date('Y-m-d');
    return date('Y-m-d', $t);
}
function h(?string $s): string { return htmlspecialchars($s ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

$sign = normalize_sign(safe_get('sign', 'Aries'));
$hdate = normalize_date(safe_get('date', '')); // Y-m-d

$english = '';
$found = false;
$err = null;

try {
    @include('../includes/connection.php'); // expects $link (mysqli)
    if (!isset($link) || !($link instanceof mysqli)) {
        throw new RuntimeException('Database link missing. Check include path.');
    }

    // Safer query: case-insensitive compare regardless of column collation
    $sql = "SELECT english_content FROM daily WHERE hdate = ? AND UPPER(sign) = UPPER(?) LIMIT 1";
    if ($stmt = $link->prepare($sql)) {
        $stmt->bind_param('ss', $hdate, $sign);
        $stmt->execute();
        $stmt->bind_result($english_content);
        if ($stmt->fetch()) {
            $english = (string)$english_content;
            $found = true;
        }
        $stmt->close();
    } else {
        // Fallback scan for the date
        $q = sprintf("SELECT sign, english_content FROM daily WHERE hdate='%s'",
            $link->real_escape_string($hdate)
        );
        if ($res = $link->query($q)) {
            while ($row = $res->fetch_assoc()) {
                if (strcasecmp((string)$row['sign'], $sign) === 0) {
                    $english = (string)($row['english_content'] ?? '');
                    $found = true;
                    break;
                }
            }
            $res->free();
        }
    }
} catch (Throwable $e) {
    $err = $e->getMessage();
}

// Presentation
$prettyDate = date('l, F j, Y', strtotime($hdate));
$signGlyphs = [
    'Aries'=>'♈︎','Taurus'=>'♉︎','Gemini'=>'♊︎','Cancer'=>'♋︎','Leo'=>'♌︎','Virgo'=>'♍︎',
    'Libra'=>'♎︎','Scorpio'=>'♏︎','Sagittarius'=>'♐︎','Capricorn'=>'♑︎','Aquarius'=>'♒︎','Pisces'=>'♓︎'
];
$glyph = $signGlyphs[$sign] ?? '★';

$allowedTags = '<p><br><b><strong><i><em><ul><ol><li><hr><span><div>';
$clean = strip_tags((string)$english, $allowedTags);
?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title><?=h($sign)?> Daily Horoscope</title>
<style>
  :root{
    --bg:#0b0b0f; --text:#f3f5f7; --muted:#a9afbf; --stroke:#23263b;
    --panel:#10121b; --accent:#8ec4ff;
  }
  html,body{height:100%}
  *{box-sizing:border-box}
  body{
    margin:0; color:var(--text); background:transparent;
    font: 15px/1.6 system-ui,-apple-system,Segoe UI,Inter,Roboto,Helvetica,Arial,sans-serif;
    -webkit-font-smoothing:antialiased; text-rendering:optimizeLegibility;
  }
  .wrap{max-width:720px; margin:0 auto; padding:12px 14px}
  .card{border:1px solid var(--stroke); border-radius:16px; padding:14px 16px;
    background:linear-gradient(180deg, rgba(255,255,255,.03), rgba(255,255,255,.02))}
  .hd{display:flex; align-items:center; justify-content:space-between; gap:10px; margin-bottom:8px}
  .title{margin:0; font-size:18px; font-weight:800; color:#fff}
  .meta{font-size:12px; color:var(--muted)}
  .signbadge{display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border:1px solid var(--stroke);
    border-radius:999px; background:#121426; font-weight:700}
  .signbadge .g{font-size:18px}
  .text{margin-top:6px}
  .text p{margin:.6em 0}
  .note{margin-top:10px; font-size:12px; color:var(--muted)}
</style>
</head>
<body>
  <div class="wrap">
    <div class="card">
      <div class="hd">
        <h1 class="title">Daily Horoscope</h1>
        <div class="signbadge"><span class="g"><?=$glyph?></span> <span><?=h($sign)?></span></div>
      </div>
      <div class="meta"><?=h($prettyDate)?></div>
      <div class="text">
        <?php if ($err): ?>
          <p><strong>Temporarily unavailable.</strong></p>
          <p class="note"><?=h($err)?></p>
        <?php elseif ($found && trim($clean) !== ''): ?>
          <?= $clean ?>
        <?php else: ?>
          <p>Horoscope for <strong><?=h($sign)?></strong> on <?=h($prettyDate)?> is not available.</p>
        <?php endif; ?>
      </div>
      <div class="note">Tip: Tap a sign tile to switch instantly.</div>
    </div>
  </div>
<script>
  function postHeight(){
    var h = document.body.scrollHeight || document.documentElement.scrollHeight;
    try{ parent.postMessage({ type: "dailyHeight", height: h }, "*"); }catch(e){}
  }
  window.addEventListener("load", postHeight);
  window.addEventListener("resize", function(){ setTimeout(postHeight, 100); });
  setTimeout(postHeight, 400);
</script>
</body>
</html>
